﻿//
// Procedural Lightning for Unity
// (c) 2015 Digital Ruby, LLC
// Source code may be used for personal or commercial projects.
// Source code may NOT be redistributed or sold.
// 

using UnityEngine;
using System.Collections;

namespace DigitalRuby.ThunderAndLightning
{
    public class LightningBeamSpellScript : LightningSpellScript
    {
        [Header("Beam")]
        [Tooltip("The lightning path script creating the beam of lightning")]
        public LightningBoltPathScriptBase LightningPathScript;

        [Tooltip("Give the end point some randomization")]
        public float EndPointRandomization = 1.5f;

        /// <summary>
        /// Callback for collision events
        /// </summary>
        [HideInInspector]
        public System.Action<RaycastHit> CollisionCallback;

        private void CheckCollision()
        {
            RaycastHit hit;

            // send out a ray to see what gets hit
            if (Physics.Raycast(SpellStart.transform.position, Direction, out hit, MaxDistance, CollisionMask))
            {
                // we hit something, set the end object position
                SpellEnd.transform.position = hit.point;

                // additional randomization of end point
                SpellEnd.transform.position += (UnityEngine.Random.insideUnitSphere * EndPointRandomization);

                // play collision sound
                PlayCollisionSound(SpellEnd.transform.position);

                // play the collision particle system
                if (CollisionParticleSystem != null)
                {
                    CollisionParticleSystem.transform.position = hit.point;
                    CollisionParticleSystem.Play();
                }

                ApplyCollisionForce(hit.point);

                // notify listeners of collisions
                if (CollisionCallback != null)
                {
                    CollisionCallback(hit);
                }
            }
            else
            {
                // stop collision particle system
                if (CollisionParticleSystem != null)
                {
                    CollisionParticleSystem.Stop();
                }

                // extend beam to max length
                SpellEnd.transform.position = SpellStart.transform.position + (Direction * MaxDistance);

                // randomize end point a bit
                SpellEnd.transform.position += (UnityEngine.Random.insideUnitSphere * EndPointRandomization);
            }
        }

        protected override void Start()
        {
            base.Start();

            LightningPathScript.ManualMode = true;
        }

        protected override void Update()
        {
            base.Update();

            if (!Casting)
            {
                return;
            }

            CheckCollision();
        }

        protected override void OnCastSpell()
        {
            LightningPathScript.ManualMode = false;
        }

        protected override void OnStopSpell()
        {
            LightningPathScript.ManualMode = true;
        }
    }
}